#include <iostream>
#include <cstdlib>
#include <bitset>
#include <iomanip>
#include <sstream>
#include <unistd.h>
#include "crc32.h"
#include "romio.h"
#include "visualize.h"

using namespace std;

//if there's only one char repeating in memory segment, return its value
//if not, return -1.
int contSameChar(char* arr, unsigned int start, unsigned int length)
{
    char e=arr[start];
    int result=arr[start];
    for (unsigned int i=start;i<start+length;i++)
    {
        if (arr[i]!=e)
        {
            result=-257;
        }
    }
    return result;
}

//does it contain the same pattern - returns minimum patern width.
//number is maximum searched pattern size
int contSamePattern(char *arr, uint32_t length, unsigned int number)
{
    unsigned int i=0;
    for (i=1;i<number;i++)
    {
    bool ok=0;
        for (unsigned int j=0;j<length;j=j+i)
        {
            for (unsigned int k=0;k<i;k++)
            {
                if (arr[k]!=arr[j+k]) ok=1;
            }
        }
    if (!ok) return i;
    }
    return 0;
}

void printUsage()
{
   cout<<"Usage: \n";
   cout<<" romcheck -m 512 -p 16 -l 4 -f 0 -x [sdf] -s [abc..] file.bin\n";
   cout<<"All parameters except file name are optional\n";
   cout<<"Parameters: \n";
   cout<<" -m X - smallest half size in comparing halves, default 512 \n";
   cout<<" -p X - largest pattern search length, default 16, 0 for no scan \n";
   cout<<" -l X - smallest literal length, default 4 \n";
   cout<<" -f X - X-character non-literal gap in literal, default 0\n";
   cout<<" -x [sdf] - exclude tests: s - stuck lines, d - duplicate halves,\n";
   cout<<"                           f - literals print out\n";
   cout<<" -s [abc..] - Characters accepted as literals. Deafult:\n";
   cout<<"   ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz\n";
   cout<<"   1234567890!()$.,? \"#_%&\'*+-/:;<=>@{}[]~\\^`\n";
   return;
}

//is nedle in haystack?
bool isInArray(char needle, char* haystack, int haystackLength)
{
    bool found=0;
    for (int i=0;i<haystackLength;i++)
    {
        if (needle==haystack[i])
        {
            found=1;
        }
    }
    return found;
}


//load file into memory

int main(int argc, char* argv[])
{
    //display information
    cout<<" ________ "<<endl;
    cout<<" |_      \\    ROM dump diagnostics and test tool "<<endl;
    cout<<" _/  /\\   \\   2013 MCbx                  v. 0.02"<<endl;
    cout<<" ___/__\\___\\  "<<endl;
    cout<<"   /    \\     To see usage run without parameters"<<endl;

    //parse arguments
    if (argc<2)
    {
        printUsage();
        return 64;
    }


    //These options can be altered with command line arguments
    bool checkStuckLines=1;  //-xs
    bool checkDuplicates=1;  //-xd
    bool checkStrings=1;     //-xt
    int minHalfSize=512;     //-m512
    int maxPatternDepth=16;  //-p16
    int literalLength=4;     //-l4
    int acceptInterleave=0;  //-f0
    char * literals;         //-s0123456789..

    //PARSE PARAMETERS
    int opt=0;
    static const char *optString = "x:m:p:l:f:s:";
    bool lit=0;
    opt = getopt( argc, argv, optString );
     while( opt != -1 )
        {
        switch( opt )
        {
        case 'm':
            minHalfSize=atoi(optarg);
            if (minHalfSize<=0)
            {
                printUsage();
                return 64;
            }
            break;
        case 'p':
            maxPatternDepth=atoi(optarg);
            if (maxPatternDepth<0)
            {
                printUsage();
                return 64;
            }
            break;
        case 'l':
            literalLength=atoi(optarg);
            if (literalLength<=0)
            {
                printUsage();
                return 64;
            }
            break;
        case 'f':
            acceptInterleave=atoi(optarg);
            if (acceptInterleave<0)
            {
                printUsage();
                return 64;
            }
            break;
        case 's':
            literals = new char[strlen(optarg)];
            strcpy(literals,optarg);
            lit=1;
            break;
        case 'x':
            char * oArgs=optarg;
            for (unsigned int i=0;i<strlen(oArgs);i++)
            {
                if (oArgs[i]=='s') checkStuckLines=0;
                if (oArgs[i]=='d') checkDuplicates=0;
                if (oArgs[i]=='t') checkStrings=0;
                if (oArgs[i]!='s' && oArgs[i]!='d' && oArgs[i]!='t')
                {
                    cout<<"\nPlease use s/d/t in -x parameter!\n";
                    printUsage();
                    return 64;
                }
            }
            break;
        }
        opt = getopt( argc, argv, optString );
     }

    //default literal table
    if (!lit)
    {
 static char ltCh[]={'0','1','2','3','4','5','6','7','8','9','!','(',')','$',
                     'a','b','c','d','e','f','g','h','i','j','k','l','m','n',
                     'o','p','q','r','s','t','u','v','w','x','y','z','A','B',
                     'C','D','E','F','G','H','I','J','K','L','M','N','O','P',
                     'Q','R','S','T','U','V','W','X','Y','Z','.',',','?','!',
                     ' ','\"','#','_','%','&','\'','*','+','-','/',':',';','<',
                     '=','>','@','{','}','[',']','~','\\','^','`', '\0'};
        literals=ltCh;
    }

    //load file to buf
    char* buffer;
    long lSize=0;
    cout<<"Loading file into memory...";
    lSize=loadRom(argv[optind],buffer,true);
    if (lSize<=0)
    {
        cout<<"\n Error when loading file!\n";
        return -65;
    }

    cout<<"Loaded consuming "<<lSize<<" bytes for data.\n";
    cout<<"Analysing file: "<<argv[optind] <<"\n";


    //Check for stuck data bits
    if (checkStuckLines){
    char stuckZeros=0x00;  //or it with each byte - should be 255.
    char stuckOnes=0x00; //

    cout<<"\nPreparing to check data lines:                D 76543210   ";
    cout<<"         D 76543210";
    cout<<"\nChecking for stuck data bits...";
    for (int i=0;i<lSize;i++)
    {
        stuckZeros|=buffer[i];
        stuckOnes|=~buffer[i];
    }
    stuckZeros=~stuckZeros;
    stuckOnes=~stuckOnes;
    std::bitset<8> x(stuckZeros);
    std::bitset<8> y(stuckOnes);
    cout<<"OK: Stuck zeros: "<<x<<", Stuck ones: "<<y<<"\n";
    if (stuckOnes!=0)
    {
        cout<<"WARNING! Stuck ones found. \n";
    }
    if (stuckZeros!=0)
    {
        cout<<"WARNING! Stuck zeros found. \n";
    }
    if ((stuckOnes==0)&&(stuckZeros==0))
    {
        cout<<"No stuck bits. \n";
    }
    }

    //Check for halves - min half size is 512 bytes
    if (checkDuplicates){
    char fault=0;
    cout<<"\nChecking for duplicated halves...\n";
    if ((minHalfSize & (minHalfSize - 1)) != 0)
    {
        cout<<"ERROR: Minimum half size is not power of 2!\n";
        return 60;
    }
    if ((lSize & (lSize - 1)) != 0)
    {
        cout<<"Size is not a power of 2 - addressing failure!\n";
        fault=1;
    }
    if (lSize<=minHalfSize)
    {
       cout<<"Size of ROM ("<<lSize<<") is smaller than size of minimal half ";
       cout<<"("<<minHalfSize<<")!\n";
        fault=1;
    }
    if (!fault)
    {
        int quant=lSize;
        do {
            quant=quant/2;
       //     cout<<"Checking Part Size: "<<quant<<": \n";
            uint32_t * hashes = new uint32_t[(lSize/quant)];
            int k=0;
            for (int i=0;i<lSize;i=i+quant)
            {
                hashes[k]=rc_crc32(0,buffer+i,quant);
                k++;
            }
            for (int i=0;i<k-1;i++)
            {
                for (int j=i;j<k+1;j++)
                {
                    if (hashes[i]==hashes[j] && i!=j)
                    {
                      fault=2;
                      cout<<"WARNING: Parts are equal: 0x"<<hex<<setfill('0');
                      cout<<setw(5)<<uppercase<<i*quant<<nouppercase<<"-0x";
                      cout<<setw(5)<<uppercase<<((i+1)*quant)-1<<nouppercase;
                      cout<<" and 0x"<<setw(5)<<uppercase<<j*quant<<nouppercase;
                      cout<<"-0x"<<setw(5)<<uppercase<<((j+1)*quant)-1;
                      cout<<nouppercase<<"\n"<<dec;
                    }
                }
            }

            for (int i=0;i<k;i++)
            {
                int same=contSameChar(buffer,i*quant,quant);
                    if (same!=-257)
                      {
                         unsigned char saa=same;
                         cout<<"WARNING: Part contains same char: 0x";
                         cout<<hex<<setfill('0')<<setw(5)<<uppercase<<i*quant;
                         cout<<nouppercase<<"-0x";
                         cout<<setw(5)<<uppercase<<((i+1)*quant)-1<<nouppercase;
                         cout<<" char: 0x"<<setw(2)<<uppercase<<(int)saa;
                         cout<<nouppercase<<"\n"<<dec;
                       }
                    int csp=contSamePattern(buffer+(i*quant),quant,maxPatternDepth);
                    if (csp>1)
                    {
                        cout<<"WARNING: Part contains repetitive pattern of ";
                        cout<<csp<<" bytes! Part: ";
                        cout<<hex<<setfill('0')<<setw(5)<<uppercase<<i*quant;
                        cout<<nouppercase<<"-0x";
                        cout<<setw(5)<<uppercase<<((i+1)*quant)-1<<nouppercase;
                        cout<<"\n";
                    }
            }

        delete [] hashes;
        } while (quant>minHalfSize);
    } else
    {
       cout<<"Not checking for duplicated halves because one or more errors.\n";
    }
    if (!fault)
    {
        cout<<"No duplicated halves found.\n";
    }
    }


    if (checkStrings) {
    //Scan for literals at least lteralLength chars width, but allowing for
    //acceptInterleave non-literal characters - this is for interlaced strings

    cout<<"\nExtracting strings of min length "<<literalLength;
    cout<<" acceptable break "<<acceptInterleave<<":\n";
    int literalCount=0;
    unsigned int literalStartPoint=0;
    unsigned int prevStartPoint;
    int prevCount = 0;
    int missed=0;
    stringstream ss;
    for (int i=0;i<lSize;i++)
    {
        if (isInArray(buffer[i],literals,strlen(literals)))
        {
            if (literalCount==0)
            {
                prevStartPoint=literalStartPoint;
                literalStartPoint=i;
            }
            literalCount++;
        }
        else
        {
            missed++;
            if (missed>acceptInterleave)
            {
                literalCount=0; //start from beginning
                missed=0;
            }
        }

        if (literalCount>=literalLength)
        {
            ss<<printHex(buffer,literalStartPoint,literalCount,lSize);
            ss<<"\n";
            prevCount=literalCount;
            literalCount=0;
        }
    }
    ss.seekp(0);    //crude patch added for bug with displaying double lines
    string prev;
    string linia;
    while (getline(ss,linia))
    {
        if (linia[0]==' ')
        {
//           cout<<". . .\n";
        }
        else
        {
            if (linia!=prev)
            {
                cout<<linia<<"\n";
                prev=linia;
            }
        }
    }
    }

    cout<<"\nEnd of analysis.\n";
    return 0;
}
